package com.payneteasy.simplecache.cassandra.impl;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.PreparedStatement;
import com.datastax.oss.driver.api.core.cql.ResultSet;
import com.datastax.oss.driver.api.core.cql.Row;
import com.payneteasy.simplecache.cassandra.ICassandraCache;
import com.payneteasy.simplecache.cassandra.ICassandraCacheMetricsListener;
import com.payneteasy.simplecache.cassandra.ICassandraCacheMetricsListener.Type;
import com.payneteasy.simplecache.cassandra.ICassandraPayloadCodec;
import com.payneteasy.simplecache.cassandra.UpsertCassandraException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;
import java.util.Optional;
import java.util.function.Function;

import static com.datastax.oss.driver.api.core.ConsistencyLevel.QUORUM;
import static com.datastax.oss.driver.api.core.ConsistencyLevel.SERIAL;
import static java.lang.System.currentTimeMillis;

public class CassandraCacheImpl<T> implements ICassandraCache<T> {

    private static final Logger LOG = LoggerFactory.getLogger(CassandraCacheImpl.class);

    private final CqlSession                session;
    private final ICassandraPayloadCodec<T> codec;
    private final PreparedStatement         insertStmt;
    private final PreparedStatement         updateStmt;
    private final PreparedStatement         upsertStmt;
    private final PreparedStatement         selectStmt;
    private final ICassandraCacheMetricsListener metricsListener;

    public CassandraCacheImpl(CqlSession session, ICassandraPayloadCodec<T> codec, PreparedStatement insertStmt, PreparedStatement updateStmt, PreparedStatement upsertStmt, PreparedStatement selectStmt, ICassandraCacheMetricsListener metricsListener) {
        this.session         = session;
        this.codec           = codec;
        this.insertStmt      = insertStmt;
        this.updateStmt      = updateStmt;
        this.upsertStmt      = upsertStmt;
        this.selectStmt      = selectStmt;
        this.metricsListener = metricsListener;
    }

    @Override
    public Optional<T> insert(String aKey, T aValue) {
        throw new IllegalStateException("Not implemented yet");
    }

    @Override
    public void upsert(String aKey, T aValue) throws UpsertCassandraException {
        OperationTimer timer = new OperationTimer();

        ByteBuffer payload = codec.encode(aValue);
        LOG.debug("Inserting key {} ...", aKey);
        ResultSet rs = session.execute(
                upsertStmt.boundStatementBuilder()
                        .setByteBuffer       (0, payload )
                        .setString           (1, aKey    )
                        .setConsistencyLevel ( QUORUM       )
                        .build()
        );

        if(!rs.wasApplied()) {
            fire(Type.UPSERT, false, timer);
            throw new UpsertCassandraException("Cannot upsert key " + aKey);
        }
        LOG.debug("Key inserted {} ...", aKey);
        fire(Type.UPSERT, true, timer);
    }

    @Override
    public boolean update(String aKey, T aValue) {
        throw new IllegalStateException("Not implemented yet");
    }

    @Override
    public Optional<T> getIfPresent(String aKey) {
        OperationTimer timer = new OperationTimer();

        ResultSet rs = session.execute(
                selectStmt.boundStatementBuilder()
                        .setString(0, aKey)
                        .setConsistencyLevel(SERIAL)
                        .build()
        );

        Row row = rs.one();
        if (row == null) {
            LOG.debug("{}: no one row", aKey);
            fire(Type.SELECT, false, timer);
            return Optional.empty();
        }

        ByteBuffer payload = row.getByteBuffer(0);
        if (payload == null) {
            LOG.debug("{}: payload is empty", aKey);
            fire(Type.SELECT, false, timer);
            return Optional.empty();
        }

        T value = codec.decode(payload);

        fire(Type.SELECT, true, timer);
        return Optional.of(value);
    }

    private void fire(Type aType, boolean aSuccess, OperationTimer aTimer) {
        metricsListener.onEvent(
                aType, aSuccess, aTimer.getSeconds()
        );
    }
}
