package com.payneteasy.simplecache.cassandra;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.cql.PreparedStatement;
import com.payneteasy.simplecache.cassandra.impl.CassandraCacheImpl;
import com.payneteasy.simplecache.cassandra.impl.CassandraCacheMetricsListenerNoOp;

import static java.util.Objects.requireNonNull;

public class CassandraCacheBuilder<T> {

    private String keyColumn;
    private String valueColumn;
    private String guardColumn;
    private String table;

    private CqlSession                session;
    private ICassandraPayloadCodec<T> codec;

    private ICassandraCacheMetricsListener metricsListener = new CassandraCacheMetricsListenerNoOp();

    public CassandraCacheBuilder<T> keyColumn(String keyColumn) {
        this.keyColumn = keyColumn;
        return this;
    }

    public CassandraCacheBuilder<T> valueColumn(String valueColumn) {
        this.valueColumn = valueColumn;
        return this;
    }

    public CassandraCacheBuilder<T> table(String table) {
        this.table = table;
        return this;
    }

    public CassandraCacheBuilder<T> session(CqlSession session) {
        this.session = session;
        return this;
    }

    public CassandraCacheBuilder<T> guardColumn(String guardColumn) {
        this.guardColumn = guardColumn;
        return this;
    }

    public CassandraCacheBuilder<T> codec(ICassandraPayloadCodec<T> codec) {
        this.codec = codec;
        return this;
    }

    public CassandraCacheBuilder<T> metrics(ICassandraCacheMetricsListener metricsListener) {
        this.metricsListener = metricsListener;
        return this;
    }

    public ICassandraCache<T> build() {

        requireNonNull(session, "No session");

        PreparedStatement select = session.prepare(
                format("select $VALUE from $TABLE where $KEY = ?")
        );

        PreparedStatement insert = session.prepare(
                format("insert into $TABLE ($KEY, $VALUE) values (?, ?) if not exists")
        );

        PreparedStatement update = session.prepare(
                format("update $TABLE set $VALUE = ? where $KEY = ? if exists")
        );

        PreparedStatement upsert = session.prepare(
                format("update $TABLE "
                        + " set   $VALUE = ?"
                        + " where $KEY   = ?"
                        + " if    $GUARD = null"
                )
        );

        return new CassandraCacheImpl<>(
                session
                , requireNonNull(codec, "No codec")
                , insert
                , update
                , upsert
                , select
                , metricsListener
        );
    }

    private String format(String aTemplate) {
        return aTemplate
                .replace("$TABLE", requireNonNull(table      , "No table"))
                .replace("$KEY"  , requireNonNull(keyColumn  , "No key column"))
                .replace("$VALUE", requireNonNull(valueColumn, "No value column"))
                .replace("$GUARD", requireNonNull(guardColumn, "No guard column"))
                ;
    }
}
